% Copyright (c) 2013, Massachusetts Institute of Technology
% This program was presented in the book "Visual Psychophysics:
% From Laboratory to Theory" by Zhong-Lin Lu and Barbara Dosher.
% The book is available at http://mitpress.mit.edu/books/visual-psychophysics

%%% Program SAT.m

function SAT (C_in_Os)

%% Display Setup Module

% Define display parameters

whichScreen = max(Screen('screens'));
p.ScreenDistance = 30; 	% in inches
p.ScreenHeight = 15; 	% in inches
p.ScreenGamma = 2;	% from monitor calibration
p.maxLuminance = 100; % from monitor calibration
p.ScreenBackground = 0.5;

% Open the display window and hide the mouse cursor

if exist('onCleanup', 'class'), oC_Obj = onCleanup(@()sca); end % close any pre-existing PTB Screen window
PsychImaging('PrepareConfiguration'); 
PsychImaging('AddTask', 'General', 'FloatingPoint32BitIfPossible');   % set up a 32-bit framebuffer
PsychImaging('AddTask', 'General', 'NormalizedHighresColorRange');
PsychImaging('AddTask', 'FinalFormatting', 'DisplayColorCorrection', 'SimpleGamma');  % setup Gamma correction method
[windowPtr p.ScreenRect] = PsychImaging('OpenWindow', whichScreen, p.ScreenBackground);  % open a display window
PsychColorCorrection('SetEncodingGamma', windowPtr, 1 / p.ScreenGamma);  % set Gamma for all color channels
HideCursor;

% Get frame rate and set screen font

p.ScreenFrameRate = FrameRate(windowPtr);
Screen('TextFont', windowPtr, 'Times'); 
Screen('TextSize', windowPtr, 24);

%% Experimental Module

% Specify the stimulus
p.stimSize = 0.98;     % letter size in visual degree
p.radius = 4.12;       % radius of annulus in degree
p.jitter = 4;          % range of location jitter in pixels
p.dispSize = [4 12];   % display sizes: total # of items in 
                       % a display
p.stimDuration = 0.1;  % letters duration in seconds
p.processTime = [0 0.05 0.15 0.30 0.50 1.15 1.80]; 
                       % cue delays
repeats = 60;          % Number of trials in each condition
p.fixDuration = 0.25;
p.feedbackDuration = 0.5;
p.ITI = 1;             % time interval between trials in 
                       % seconds

% Compute stimulus parameters
ppd = pi/180 * p.ScreenDistance / p.ScreenHeight * ...
      p.ScreenRect(4);
m = round(p.stimSize * ppd);  % letter size in pixels
fixXY = [[[-1 1] * m / 2 0 0] + p.ScreenRect(3) / 2;
         [0 0 [-1 1] * m / 2] + p.ScreenRect(4) / 2];
nDispSize = length(p.dispSize);
nProcessTime = length(p.processTime);
nTrials = repeats * nDispSize * nProcessTime * 2; 
                       % total number of trials
p.randSeed = ClockRandSeed;  % use clock to set the seed of
                             % the random number generator
radius = p.radius * ppd;     % radius in pixels
theta = (0 : 360/15 : 359)'; % polar angles of the 15 
                             % locations
stimDur = (round(p.stimDuration * p.ScreenFrameRate) ...
      - 0.5) / p.ScreenFrameRate ;
[x y] = meshgrid(linspace(-1, 1, m));
r = sqrt(x .^ 2  + y .^ 2);
circle = (1 - exp(-((r - 0.85) / 0.15) .^ 4)) * ...
      p.ScreenBackground;    % circle with blurred edge
texB = Screen('MakeTexture', windowPtr, ones(m) * ...
      p.ScreenBackground, 0, 0, 2);   % background
texD = Screen('MakeTexture', windowPtr, circle, 0, 0, 2); 
                       % distractors
texT = Screen('MakeTexture', windowPtr, circle, 0, 0, 2); 
                       % target
tex = texD;

if nargin && C_in_Os, tex = texT; end
Screen('FillRect', tex, p.ScreenBackground, [m/2 m/4 ...
      m m/4*3]); % C open to right

% Initialize a table to set up experimental conditions
p.recLabel = {'trialIndex' 'sizeIndex' 'processTimeIndex' ...
      'targetPresent' 'respCorrect' 'respTime'};
rec = nan(nTrials, length(p.recLabel)); 
        % matrix rec initialized with NaN
rec(:, 1) = 1 : nTrials; 
        % count the trial numbers from 1 to nTrials
sizeIndex = repmat(1 : nDispSize, [2 1 repeats ...
      nProcessTime]);
targetPresent = zeros(2, nDispSize, repeats, nProcessTime); 
        % first set all to 0
targetPresent(:, :, 1 : repeats / 2, :) = 1; 
        % change first half to 1
timeIndex = repmat(1 : nProcessTime, [2 nDispSize ...
      repeats 1]);
[rec(:, 2) ind] = Shuffle(sizeIndex(:)); 
        % shuffle size index
rec(:, 3) = timeIndex(ind); 
        % shuffle process time index in the same order
rec(:, 4) = targetPresent(ind); 
        % shuffle target presence index in the same order

% Prioritize display to optimize display timing
Priority(MaxPriority(windowPtr));

str = ['Press the left buttons for target absent and the ' ...
       'right buttons for target present responses\n\n' ...
       'Please respond as quickly and accurately as ' ...
       'possible.\n\n''Press any button to start.'];
DrawFormattedText(windowPtr, str, 'center', 'center', 1);
        % Draw Instruction text string centered in window
Screen('Flip', windowPtr);  
        % flip the text image into active buffer
Beeper;

% Initialize RTBox
RTBox('ButtonNames', {'left' 'left' 'right' 'right'}); 
        % define the first two buttons as left; the last 
        % two as right.
RTBox(inf);           % wait till any button is pressed
Secs = Screen('Flip', windowPtr);
p.start = datestr(now); % record start time

% Run nTrials trials
for i = 1 : nTrials
        % parameters for this trial
    sz = p.dispSize(rec(i, 2)); 
        % display size: total # of items in the display
    angles = theta + rand * 360;
    x = radius * cosd(angles) + p.ScreenRect(3) / 2; 
        % center of rec for item locations
    y = radius * sind(angles) + p.ScreenRect(4) / 2;
    x = x + (rand(15, 1) - 0.5) * 2 * p.jitter; 
        % apply jitter
    y = y + (rand(15, 1) - 0.5) * 2 * p.jitter;
    rects = CenterRectOnPoint([0 0 m m], x, y); % 15 rects
    dt = p.processTime(rec(i, 3));
    nLoc = sz + sz / 4 - 1;
    tex(1 : nLoc) = texD;   
        % set up distractors
    tex([5 10 nLoc+1:15]) = texB; 
        % 5, 10 and higher ones set as background
    if rec(i, 4)     % target present
        ind = randi(nLoc);
        if mod(ind, 5) == 0, ind = ind + 1; end 
                     % avoid blank location
        tex(ind) = texT;
    end

    WaitTill(Secs + p.ITI);
    RTBox('clear'); % clear RTBox, sync clocks
    Screen('DrawLines', windowPtr, fixXY, 3, 0); % fixation
    t0 = Screen('Flip', windowPtr, 0, 1);
    Screen('DrawTextures', windowPtr, tex, [], rects'); 
                     % C and O
    t0 = Screen('Flip', windowPtr, t0 + p.fixDuration);
    t0 = Screen('Flip', windowPtr, t0 + stimDur); 
                     % turn off stim
    tCue = WaitSecs('UntilTime', t0 + dt);
    Beeper;          % please double check if there is any 
                     % delay introdued by your operating
                     % system
    RTBox('clear', 0); % clear any response before the tone
    [Secs key] = RTBox(inf);   % wait till response
    if iscellstr(key), key = key{1}; end 
        % take the first in case of multiple key presses
    rec(i, 5) = strcmp(key, 'right') == rec(i, 4); 
        % record correctness
    rec(i, 6) = Secs(1) - t0; % record respTime
    str = sprintf('%.0f', (Secs(1) - tCue) * 1000);
    DrawFormattedText(windowPtr, str, 'center', 'center', ...
        1);
    t0 = Screen('Flip', windowPtr); % show feed back
    Screen('Flip', windowPtr, t0 + p.feedbackDuration); 
        % turn off feedback
end
p.finish = datestr(now); % record finish time
save SAT_rst.mat rec p;  % save the results


%% System Reinstatement Module

Priority(0);  % restore priority
sca; % close window and textures, restore color lookup table


